// Popup logic for Maps Leads Scraper

const els = {
  searches: document.getElementById('searches'),
  locations: document.getElementById('locations'),
  start: document.getElementById('startBtn'),
  stop: document.getElementById('stopBtn'),
  exportCsv: document.getElementById('exportCsvBtn'),
  rows: document.getElementById('rows'),
  filterType: document.getElementById('filterType'),
  filterQuery: document.getElementById('filterQuery'),
  clearAll: document.getElementById('clearAll'),
  filterEmail: document.getElementById('filterEmail'),
  filterWebsite: document.getElementById('filterWebsite'),
  filterPhone: document.getElementById('filterPhone'),
  dedup: document.getElementById('dedup'),
  countName: document.getElementById('countName'),
  countPhone: document.getElementById('countPhone'),
  countWebsite: document.getElementById('countWebsite'),
  countEmail: document.getElementById('countEmail'),
  countLocation: document.getElementById('countLocation'),
  avatarBtn: document.getElementById('avatarBtn'),
  avatarMenu: document.getElementById('avatarMenu'),
  avatarName: document.getElementById('avatarName'),
  menuUsername: document.getElementById('menuUsername'),
  menuStatus: document.getElementById('menuStatus'),
  menuRemaining: document.getElementById('menuRemaining'),
  menuExpiry: document.getElementById('menuExpiry'),
  logoutBtn: document.getElementById('logoutBtn'),
};

let data = []; // {Name, Phone number, Website, Email, Location}
let originalData = [];
let isRunning = false; // UI hint; real running lives in background
const seenKeys = new Set();

function setStopButton(mode){ // mode: 'stop' | 'resume' | 'idle'
  try {
    if (!els.stop) return;
    const btn = els.stop;
    btn.classList.remove('ghost','btn-danger','primary');
    if (!btn.classList.contains('btn-icon')) btn.classList.add('btn-icon');
    if (mode === 'stop') {
      btn.classList.add('btn-danger');
      btn.dataset.mode = 'stop';
      btn.innerHTML = '<i class="bi bi-stop-fill"></i><span>Arrêter</span>';
    } else if (mode === 'resume') {
      btn.classList.add('primary');
      btn.dataset.mode = 'resume';
      btn.innerHTML = '<i class="bi bi-play-fill"></i><span>Reprendre</span>';
    } else {
      btn.classList.add('ghost');
      btn.dataset.mode = 'idle';
      btn.innerHTML = '<i class="bi bi-stop-fill"></i><span>Arrêter</span>';
    }
  } catch {}
}

function cleanPhone(v){
  try {
    let s = String(v||'');
    s = s.replace(/%20/gi, '');
    s = s.replace(/\D+/g, '');
    return s;
  } catch(_) { return v; }
}
function cleanEmail(v){
  try {
    let s = String(v||'').trim();
    s = s.replace(/%20/gi, '');
    s = s.split(/[?#]/)[0];
    s = s.replace(/(\.(png|jpg|jpeg|gif|bmp|webp|tiff|svg|ico|heic|heif))+$/i, '');
    return s;
  } catch(_) { return v; }
}
function isValidEmail(v){
  try { return /^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}$/.test(String(v||'').trim()); } catch(_) { return false; }
}

function renderRows(rows){
  els.rows.innerHTML = '';
  rows.forEach((row, idx) => {
    const el = document.createElement('div');
    el.className = 'row';
    el.innerHTML = `
      <div class="cell"><input value="${escapeHtml(row['Name'])}" data-k="Name" data-i="${idx}"></div>
      <div class="cell"><input value="${escapeHtml(row['Phone number'])}" data-k="Phone number" data-i="${idx}"></div>
      <div class="cell"><input value="${escapeHtml(row['Website'])}" data-k="Website" data-i="${idx}"></div>
      <div class="cell"><input value="${escapeHtml(row['Email'])}" data-k="Email" data-i="${idx}"></div>
      <div class="cell"><input value="${escapeHtml(row['Location'])}" data-k="Location" data-i="${idx}"></div>
      `;
    els.rows.appendChild(el);
  });
}

function updateCounts(rows){
  try {
    const countAvail = (key) => rows.reduce((acc, r) => {
      const v = String(r[key] ?? '').trim();
      if (v && v.toLowerCase() !== 'not available') return acc + 1;
      return acc;
    }, 0);
    if (els.countName) els.countName.textContent = `(${countAvail('Name')})`;
    if (els.countPhone) els.countPhone.textContent = `(${countAvail('Phone number')})`;
    if (els.countWebsite) els.countWebsite.textContent = `(${countAvail('Website')})`;
    if (els.countEmail) els.countEmail.textContent = `(${countAvail('Email')})`;
    if (els.countLocation) els.countLocation.textContent = `(${countAvail('Location')})`;
  } catch {}
}

function populateAvatar(auth){
  try {
    const uname = auth?.username || 'Compte';
    if (els.avatarName) els.avatarName.textContent = uname;
    if (els.menuUsername) els.menuUsername.textContent = uname;
    if (els.menuStatus) {
      const st = (auth?.status || '').toString();
      els.menuStatus.textContent = st ? (st.charAt(0).toUpperCase() + st.slice(1).toLowerCase()) : '-';
    }
    if (els.menuRemaining) {
      const rd = (auth?.remainingDays ?? auth?.accountAge ?? null);
      els.menuRemaining.textContent = rd == null ? '-' : `${rd} jours restants`;
    }
    if (els.menuExpiry) {
      const exp = auth?.expirationDate ? new Date(auth.expirationDate) : null;
      els.menuExpiry.textContent = exp ? exp.toLocaleDateString() : '-';
    }
  } catch {}
}

function applyFilters(base){
  let rows = [...base];
  if (els.filterEmail.checked) rows = rows.filter(r => isValidEmail(r['Email']));
  if (els.filterWebsite.checked) rows = rows.filter(r => r['Website'] && r['Website'] !== 'Not available');
  if (els.filterPhone.checked) rows = rows.filter(r => r['Phone number'] && r['Phone number'] !== 'Not available');
  if (els.dedup.checked) {
    const seen = new Set();
    rows = rows.filter(r => {
      const key = `${r['Name'].trim().toLowerCase()}|${String(r['Phone number']).trim()}|${String(r['Website']).trim()}`;
      if (seen.has(key)) return false; seen.add(key); return true;
    });
  }

  const type = els.filterType.value.toLowerCase();
  const q = (els.filterQuery.value || '').trim().toLowerCase();
  if (q) {
    rows = rows.filter(r => {
      if (type === 'location') return String(r['Location']).trim().toLowerCase().includes(q);
      if (type === 'email') return String(r['Email']).trim().toLowerCase().includes(q);
      if (type === 'website') return String(r['Website']).trim().toLowerCase().includes(q);
      if (type === 'name') return String(r['Name']).trim().toLowerCase().includes(q);
      if (type === 'phone') {
        const p = String(r['Phone number']);
        return p.startsWith(q.slice(0,2)) && p.slice(0, q.length) === q;
      }
      return true;
    });
  }
  return rows;
}

function persistSettings(){
  const settings = {
    filterEmail: !!(els.filterEmail && els.filterEmail.checked),
    filterWebsite: !!(els.filterWebsite && els.filterWebsite.checked),
    filterPhone: !!(els.filterPhone && els.filterPhone.checked),
    dedup: !!(els.dedup && els.dedup.checked),
  };
  chrome.storage.local.set({ mls_settings: settings }).catch(()=>{});
}

function refresh(){
  data = originalData.map(d => ({...d, 'Phone number': cleanPhone(d['Phone number']), 'Email': cleanEmail(d['Email']) }));
  const rows = applyFilters(data);
  renderRows(rows);
  updateCounts(rows);
}

function escapeHtml(s){
  return String(s).replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c]));
}

function toCSV(rows){
  const headers = ['Name','Phone number','Website','Email','Location'];
  const lines = [headers.join(',')];
  for (const r of rows){
    const vals = headers.map(h => {
      const v = String(r[h] ?? '');
      const needsQuote = /[",\n]/.test(v);
      return needsQuote ? '"' + v.replace(/"/g,'""') + '"' : v;
    });
    lines.push(vals.join(','));
  }
  return lines.join('\n');
}

async function ensureMapsTab(){
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  const active = tabs[0];
  const isMaps = active && /https?:\/\/(www\.)?google\.(com|\w+)\/maps\//.test(active.url || '');
  if (isMaps) return active.id;
  const created = await chrome.tabs.create({ url: 'https://www.google.com/maps', active: true });
  await waitForTabComplete(created.id, 20000);
  return created.id;
}

async function sendToContent(tabId, msg){
  return chrome.tabs.sendMessage(tabId, msg).catch(()=>({ ok:false }));
}

function sanitizeList(list){
  return list.map(x => String(x||'').trim()).filter(Boolean);
}

async function runScrape(){
  isRunning = true; originalData = []; refresh();
  const searches = sanitizeList((els.searches.value || '').split(/\n+/));
  const locations = sanitizeList((els.locations.value || '').split(/\n+/));
  for (const qs of searches){
    for (const loc of locations){
      if (!isRunning) break;
      const tabId = await ensureMapsTab();
      await chrome.tabs.update(tabId, { url: `https://www.google.com/maps/search/${encodeURIComponent(qs + ' ' + loc)}?hl=en` });
      await waitForTabComplete(tabId, 30000);
      const ready = await waitForContentReady(tabId, 15000);
      if (!ready) continue;
      await sendToContent(tabId, { type: 'mls_start', location: loc });
      const done = await waitForDone(15000);
      if (!done) {
        // second pass attempt
        await sendToContent(tabId, { type: 'mls_start', location: loc });
        await waitForDone(8000);
      }
      await waitMs(400); // small pacing between locations
    }
    await waitMs(600); // pacing between different search terms
  }
  isRunning = false;
}

function waitMs(ms){ return new Promise(r => setTimeout(r, ms)); }

function waitForDone(timeoutMs){
  return new Promise(resolve => {
    let done = false;
    const timer = setTimeout(() => { if (!done) resolve(false); }, timeoutMs);
    function onMsg(msg){
      if (msg?.type === 'mls_done') { done = true; cleanup(); resolve(true); }
    }
    function cleanup(){ chrome.runtime.onMessage.removeListener(onMsg); clearTimeout(timer); }
    chrome.runtime.onMessage.addListener(onMsg);
  });
}

function onItem(row){
  const key = `${(row['Name']||'').trim().toLowerCase()}|${String(row['Phone number']||'').trim()}|${String(row['Website']||'').trim()}|${String(row['Location']||'').trim().toLowerCase()}`;
  if (seenKeys.has(key)) return;
  seenKeys.add(key);
  originalData.push(row);
  refresh();
}

// Handlers
els.start.addEventListener('click', async () => {
  if (isRunning) return;
  isRunning = true;
  const searches = sanitizeList((els.searches.value || '').split(/\n+/));
  const locations = sanitizeList((els.locations.value || '').split(/\n+/));
  // Ask background to run job so it persists if popup closes
  await chrome.runtime.sendMessage({ type: 'mls_begin_job', searches, locations });
  setStopButton('stop');
});
els.stop.addEventListener('click', async () => {
  const mode = els.stop?.dataset.mode || 'idle';
  if (mode === 'stop') {
    // Pause job
    await chrome.runtime.sendMessage({ type: 'mls_pause_job' });
    isRunning = false;
    setStopButton('resume');
  } else if (mode === 'resume') {
    // Resume job
    await chrome.runtime.sendMessage({ type: 'mls_resume_job' });
    isRunning = true;
    setStopButton('stop');
  } else {
    // Idle -> nothing
  }
});
els.exportCsv.addEventListener('click', () => {
  const rows = applyFilters(data);
  const csv = toCSV(rows);
  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8' });
  const url = URL.createObjectURL(blob);
  chrome.downloads.download({ url, filename: `maps_leads_${Date.now()}.csv`, saveAs: true });
});
// Export Excel using XLSX (SheetJS mini build included as vendor/xlsx.mini.min.js)
const exportXlsBtn = document.getElementById('exportXlsBtn');
if (exportXlsBtn) exportXlsBtn.addEventListener('click', () => {
  try {
    const headers = ['Name','Phone number','Website','Email','Location'];
    const rows = applyFilters(data);
    const aoa = [headers, ...rows.map(r => headers.map(h => String(r[h] ?? '')) )];
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.aoa_to_sheet(aoa);
    XLSX.utils.book_append_sheet(wb, ws, 'Sheet1');
    const out = XLSX.write(wb, { bookType: 'xlsx', type: 'array' });
    const blob = new Blob([out], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
    const url = URL.createObjectURL(blob);
    chrome.downloads.download({ url, filename: `maps_leads_${Date.now()}.xlsx`, saveAs: true });
  } catch (e) {
    console.error("Échec de l'export XLSX", e);
  }
});
// Avatar dropdown interactions
if (els.avatarBtn) els.avatarBtn.addEventListener('click', (e) => {
  e.stopPropagation();
  els.avatarMenu?.classList.toggle('open');
});
document.addEventListener('click', (e) => {
  try {
    if (!els.avatarMenu) return;
    const within = e.target === els.avatarMenu || els.avatarMenu.contains(e.target) || e.target === els.avatarBtn || els.avatarBtn.contains(e.target);
    if (!within) els.avatarMenu.classList.remove('open');
  } catch {}
});
if (els.logoutBtn) els.logoutBtn.addEventListener('click', async () => {
  try { await chrome.storage.local.remove(['mls_auth']); } catch {}
  try { window.location.href = 'auth.html'; } catch {}
});
// Apply filter on Enter inside the filter input
els.filterQuery.addEventListener('keydown', (e) => {
  if (e.key === 'Enter') { e.preventDefault(); refresh(); }
});
// Optional: auto-refresh when filter type or toggles change
els.filterType.addEventListener('change', refresh);
if (els.filterEmail) els.filterEmail.addEventListener('change', () => { persistSettings(); refresh(); });
if (els.filterWebsite) els.filterWebsite.addEventListener('change', () => { persistSettings(); refresh(); });
if (els.filterPhone) els.filterPhone.addEventListener('change', () => { persistSettings(); refresh(); });
if (els.dedup) els.dedup.addEventListener('change', () => { persistSettings(); refresh(); });
// Clear all data button
if (els.clearAll) els.clearAll.addEventListener('click', async () => {
  originalData = [];
  data = [];
  seenKeys.clear();
  try { await chrome.storage.local.remove(['mls_data']); } catch {}
  refresh();
  setStopButton('idle');
});
els.rows.addEventListener('input', (e) => {
  const t = e.target; if (!t.matches('input[data-k]')) return;
  const i = Number(t.dataset.i), k = t.dataset.k;
  if (!Number.isFinite(i) || !k) return;
  const rows = applyFilters(data);
  const row = rows[i]; if (!row) return;
  row[k] = k === 'Phone number' ? cleanPhone(t.value) : (k === 'Email' ? cleanEmail(t.value) : t.value);
  const idx = originalData.findIndex(r => r === row);
  if (idx >= 0) originalData[idx] = row;
});


// Init
(async function init(){
  const st = await chrome.storage.local.get(['mls_settings','mls_inputs']);
  if (st.mls_inputs){
    els.searches.value = st.mls_inputs.searches || '';
    els.locations.value = st.mls_inputs.locations || '';
  }
  if (st.mls_settings){
    if (els.filterEmail) els.filterEmail.checked = !!st.mls_settings.filterEmail;
    if (els.filterWebsite) els.filterWebsite.checked = !!st.mls_settings.filterWebsite;
    if (els.filterPhone) els.filterPhone.checked = !!st.mls_settings.filterPhone;
    if (els.dedup) els.dedup.checked = !!st.mls_settings.dedup;
  }
  // load any persisted data from background
  const d = await chrome.storage.local.get(['mls_data']);
  if (Array.isArray(d.mls_data)) {
    originalData = d.mls_data.slice();
    for (const r of originalData) {
      const key = `${(r['Name']||'').trim().toLowerCase()}|${String(r['Phone number']||'').trim()}|${String(r['Website']||'').trim()}|${String(r['Location']||'').trim().toLowerCase()}`;
      seenKeys.add(key);
    }
    refresh();
  }
  // Populate avatar from stored auth or redirect to login if missing
  try {
    const a = await chrome.storage.local.get(['mls_auth']);
    const auth = a.mls_auth || {};
    if (!auth.token) {
      try { window.location.href = 'auth.html'; } catch {}
    } else {
      populateAvatar(auth);
    }
  } catch {}
  // Sync stop button with background status
  try {
    const st2 = await chrome.runtime.sendMessage({ type: 'mls_get_status' });
    if (st2 && st2.ok) {
      if (st2.status.running) { isRunning = true; setStopButton('stop'); }
      else if ((st2.status.queueLength||0) > 0) { isRunning = false; setStopButton('resume'); }
      else { setStopButton('idle'); }
    } else { setStopButton('idle'); }
  } catch { setStopButton('idle'); }
  ['searches','locations'].forEach(k => {
    els[k].addEventListener('input', throttle(() => {
      chrome.storage.local.set({ mls_inputs: { searches: els.searches.value, locations: els.locations.value } });
    }, 400));
  });
  // Global real-time stream listener
  chrome.runtime.onMessage.addListener((msg) => {
    if (msg?.type === 'mls_item' && msg.data) onItem(msg.data);
    if (msg?.type === 'mls_done') { isRunning = false; setStopButton('idle'); }
  });
})();

function throttle(fn, wait){
  let t = 0, lastArgs = null;
  return function(...args){
    lastArgs = args;
    const now = Date.now();
    if (now - t > wait){ t = now; fn.apply(this, lastArgs); lastArgs = null; }
    else {
      clearTimeout(throttle._timer);
      throttle._timer = setTimeout(() => { t = Date.now(); fn.apply(this, lastArgs); lastArgs = null; }, wait);
    }
  };
}

function waitForTabComplete(tabId, timeoutMs = 20000){
  const start = Date.now();
  return new Promise(resolve => {
    const check = async () => {
      try {
        const [t] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (t && t.id === tabId && t.status === 'complete') return resolve(true);
      } catch {}
      if (Date.now() - start > timeoutMs) return resolve(false);
      setTimeout(check, 200);
    };
    check();
  });
}

async function waitForContentReady(tabId, timeoutMs = 10000){
  const start = Date.now();
  while (Date.now() - start < timeoutMs){
    const res = await sendToContent(tabId, { type: 'mls_ping' });
    if (res && res.ok && res.ready) return true;
    await waitMs(300);
  }
  return false;
}
