param(
    [Parameter(Position=0)]
    [string]$TargetDir = "."
)

Set-StrictMode -Version Latest
$ErrorActionPreference = 'Stop'

function New-BackupDirectory {
    param([string]$Dir)
    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    $parent = Split-Path -Parent (Resolve-Path $Dir)
    $name = Split-Path -Leaf (Resolve-Path $Dir)
    $backup = Join-Path $parent ("${name}_backup_$timestamp")
    Copy-Item -Path (Resolve-Path $Dir) -Destination $backup -Recurse -Force
    return $backup
}

function Convert-ToBase64Literal {
    param([string]$s)
    $bytes = [System.Text.Encoding]::UTF8.GetBytes($s)
    return [System.Convert]::ToBase64String($bytes)
}

function Process-JSContent {
    param([string]$code)

    $sb = New-Object System.Text.StringBuilder

    $i = 0
    $len = $code.Length

    $IN_STRING = $false
    $STRING_QUOTE = ''
    $IN_TEMPLATE = $false
    $IN_LINE_COMMENT = $false
    $IN_BLOCK_COMMENT = $false
    $ESCAPE_NEXT = $false
    $IN_REGEX = $false

    # Helper to decide if a '/' starts a regex literal (very heuristic)
    function Is-RegexStart([string]$acc) {
        if ($acc.Length -eq 0) { return $true }
        $ch = $acc[$acc.Length-1]
        # After these tokens, a regex can appear
        return "(=:+-*,!%&|^~?{[;\n".IndexOf($ch) -ge 0
    }

    # First pass: remove comments safely
    $out = New-Object System.Text.StringBuilder
    while ($i -lt $len) {
        $c = $code[$i]
        $n = if ($i + 1 -lt $len) { $code[$i+1] } else { [char]0 }

        if ($IN_LINE_COMMENT) {
            if ($c -eq "`n" -or $c -eq "`r") {
                $IN_LINE_COMMENT = $false
                [void]$out.Append($c)
            }
            $i++
            continue
        }
        if ($IN_BLOCK_COMMENT) {
            if ($c -eq '*' -and $n -eq '/') {
                $IN_BLOCK_COMMENT = $false
                $i += 2
                continue
            }
            $i++
            continue
        }
        if ($IN_STRING) {
            [void]$out.Append($c)
            if (-not $ESCAPE_NEXT -and $c -eq $STRING_QUOTE) {
                $IN_STRING = $false
            }
            $ESCAPE_NEXT = (-not $ESCAPE_NEXT -and $c -eq '\\')
            $i++
            continue
        }
        if ($IN_TEMPLATE) {
            [void]$out.Append($c)
            if ($c -eq '`' -and -not $ESCAPE_NEXT) { $IN_TEMPLATE = $false }
            $ESCAPE_NEXT = (-not $ESCAPE_NEXT -and $c -eq '\\')
            $i++
            continue
        }
        if ($IN_REGEX) {
            [void]$out.Append($c)
            if (-not $ESCAPE_NEXT -and $c -eq '/') { $IN_REGEX = $false }
            $ESCAPE_NEXT = (-not $ESCAPE_NEXT -and $c -eq '\\')
            $i++
            continue
        }

        # detect start of string/template
        if ($c -eq '"' -or $c -eq "'") {
            $IN_STRING = $true
            $STRING_QUOTE = $c
            [void]$out.Append($c)
            $i++
            continue
        }
        if ($c -eq '`') {
            $IN_TEMPLATE = $true
            [void]$out.Append($c)
            $i++
            continue
        }

        # detect comments vs divide/regex
        if ($c -eq '/' -and $n -eq '/') {
            $IN_LINE_COMMENT = $true
            $i += 2
            continue
        }
        if ($c -eq '/' -and $n -eq '*') {
            $IN_BLOCK_COMMENT = $true
            $i += 2
            continue
        }
        if ($c -eq '/') {
            # heuristic: regex start
            $prefix = $out.ToString()
            if (Is-RegexStart $prefix) {
                $IN_REGEX = $true
            }
            [void]$out.Append($c)
            $i++
            continue
        }

        [void]$out.Append($c)
        $i++
    }

    $codeNoComments = $out.ToString()

    # Second pass: encode string literals and compact whitespace (outside literals/templates/regex)
    $i = 0
    $len = $codeNoComments.Length
    $IN_STRING = $false
    $STRING_QUOTE = ''
    $IN_TEMPLATE = $false
    $IN_REGEX = $false
    $ESCAPE_NEXT = $false
    $acc = New-Object System.Text.StringBuilder

    function Append-WSCompact([string]$ch) {
        # Collapse multiple whitespace to single space, remove around punctuation
        $last = if ($acc.Length -gt 0) { $acc[$acc.Length-1] } else { [char]0 }
        $isWS = [char]::IsWhiteSpace($ch)
        if ($isWS) {
            if ($last -ne ' ' -and $last -ne "`n" -and $last -ne "`r") {
                [void]$acc.Append(' ')
            }
        } else {
            [void]$acc.Append($ch)
        }
    }

    while ($i -lt $len) {
        $c = $codeNoComments[$i]
        $n = if ($i + 1 -lt $len) { $codeNoComments[$i+1] } else { [char]0 }

        if ($IN_STRING) {
            if (-not $ESCAPE_NEXT -and $c -eq $STRING_QUOTE) {
                $IN_STRING = $false
                # close string, replace with decoder
                $literal = $sb.ToString()
                $sb.Clear() | Out-Null
                $b64 = Convert-ToBase64Literal -s $literal
                [void]$acc.Append("__s('$b64')")
                $i++
                continue
            }
            if (-not $ESCAPE_NEXT -and $c -eq '\\') { $ESCAPE_NEXT = $true } else { $ESCAPE_NEXT = $false }
            [void]$sb.Append($c)
            $i++
            continue
        }
        if ($IN_TEMPLATE) {
            [void]$acc.Append($c)
            if (-not $ESCAPE_NEXT -and $c -eq '`') { $IN_TEMPLATE = $false }
            if (-not $ESCAPE_NEXT -and $c -eq '\\') { $ESCAPE_NEXT = $true } else { $ESCAPE_NEXT = $false }
            $i++
            continue
        }
        if ($IN_REGEX) {
            [void]$acc.Append($c)
            if (-not $ESCAPE_NEXT -and $c -eq '/') { $IN_REGEX = $false }
            if (-not $ESCAPE_NEXT -and $c -eq '\\') { $ESCAPE_NEXT = $true } else { $ESCAPE_NEXT = $false }
            $i++
            continue
        }

        if ($c -eq '"' -or $c -eq "'") {
            $IN_STRING = $true
            $STRING_QUOTE = $c
            $sb.Clear() | Out-Null
            $i++
            continue
        }
        if ($c -eq '`') {
            $IN_TEMPLATE = $true
            [void]$acc.Append($c)
            $i++
            continue
        }
        if ($c -eq '/') {
            # might be regex start
            $before = $acc.ToString()
            if (Is-RegexStart $before) { $IN_REGEX = $true }
            [void]$acc.Append($c)
            $i++
            continue
        }

        if ([char]::IsWhiteSpace($c)) {
            Append-WSCompact $c
        } else {
            [void]$acc.Append($c)
        }
        $i++
    }

    $body = $acc.ToString()

    # Remove spaces around common punctuations for a bit more compaction
    $body = $body -replace "\s*([=,:;\{\}\(\)\[\]\+\-\*/<>\|&\!\?])\s*", '$1'

    # Prepend runtime decoder and wrap in IIFE without eval
    $prolog = "(function(){'use strict';function __s(b){if(typeof atob==='function'){return decodeURIComponent(escape(atob(b)));}var i,c,out='',tbl='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';var p=0;while(p<b.length){i=(tbl.indexOf(b[p++])<<18)|(tbl.indexOf(b[p++])<<12)|(tbl.indexOf(b[p++])<<6)|tbl.indexOf(b[p++]);out+=String.fromCharCode((i>>16)&255,(i>>8)&255,i&255);}try{return decodeURIComponent(escape(out.replace(/\0+$/,'')));}catch(e){return out;}}"
    $epilog = "})();"

    return $prolog + $body + $epilog
}

function Process-JSFile {
    param([string]$path)
    Write-Host "Obfuscating: $path"
    $code = Get-Content -LiteralPath $path -Raw -Encoding UTF8
    $processed = Process-JSContent -code $code
    [System.IO.File]::WriteAllText($path, $processed, [System.Text.Encoding]::UTF8)
}

$root = Resolve-Path $TargetDir
Write-Host "Target: $root"
$backup = New-BackupDirectory -Dir $root
Write-Host "Backup created at: $backup"

$jsFiles = Get-ChildItem -LiteralPath $root -Recurse -Include *.js -File | Where-Object { $_.Name -notmatch '\.min\.js$' }

foreach ($f in $jsFiles) {
    Process-JSFile -path $f.FullName
}

Write-Host "Done. Processed $($jsFiles.Count) files."
