import { promises as fs } from 'fs';
import path from 'path';
import fse from 'fs-extra';
import { globSync } from 'glob';
import { minify as htmlMinify } from 'html-minifier-terser';
import CleanCSS from 'clean-css';
import JavaScriptObfuscator from 'javascript-obfuscator';
import { minify as terserMinify } from 'terser';

const root = process.cwd();
const srcDir = root; // current folder contains source files
const distDir = path.join(root, 'dist');

const htmlOptions = {
  collapseWhitespace: true,
  removeComments: true,
  removeRedundantAttributes: true,
  removeEmptyAttributes: true,
  minifyCSS: true,
  minifyJS: true,
};

const obfuscateOptions = {
  compact: true,
  controlFlowFlattening: true,
  controlFlowFlatteningThreshold: 0.75,
  deadCodeInjection: true,
  deadCodeInjectionThreshold: 0.2,
  disableConsoleOutput: true,
  identifierNamesGenerator: 'hexadecimal',
  identifiersPrefix: 'mls_',
  numbersToExpressions: true,
  selfDefending: true,
  simplify: true,
  splitStrings: true,
  stringArray: true,
  stringArrayThreshold: 0.75,
  rotateStringArray: true,
  stringArrayEncoding: ['base64'],
  transformObjectKeys: true,
  target: 'browser',
  // Ensure Chrome APIs stay intact
  reservedNames: [
    '^chrome$', '^chrome\\..*',
    '^window$', '^document$',
    '^fetch$', '^crypto$',
  ],
};

const terserOptions = {
  compress: { passes: 2, drop_console: true },
  mangle: { toplevel: true, safari10: true },
  format: { comments: false },
  toplevel: true,
};

async function ensureCleanDist(){
  await fse.remove(distDir);
  await fse.ensureDir(distDir);
}

function isSourceFile(p){
  const base = path.basename(p);
  if (base.startsWith('dist')) return false;
  return ['.html', '.css', '.js', '.json', '.png', '.jpg', '.jpeg', '.gif', '.svg', '.webp', '.ico'].some(ext => base.endsWith(ext));
}

async function copyStatic(){
  // Copy manifest and assets as-is first
  const assets = ['manifest.json', 'logo.png'];
  for (const name of assets){
    const src = path.join(srcDir, name);
    if (await fse.pathExists(src)){
      await fse.copy(src, path.join(distDir, name));
    }
  }
}

async function processHTML(){
  const files = globSync('*.html', { cwd: srcDir });
  for (const file of files){
    const input = await fs.readFile(path.join(srcDir, file), 'utf8');
    const out = await htmlMinify(input, htmlOptions);
    await fs.writeFile(path.join(distDir, file), out, 'utf8');
  }
}

async function processCSS(){
  const files = globSync('*.css', { cwd: srcDir });
  const cleaner = new CleanCSS({ level: 2 });
  for (const file of files){
    const input = await fs.readFile(path.join(srcDir, file), 'utf8');
    const out = cleaner.minify(input).styles;
    await fs.writeFile(path.join(distDir, file), out, 'utf8');
  }
}

async function processJS(){
  const files = globSync('*.js', { cwd: srcDir });
  for (const file of files){
    const input = await fs.readFile(path.join(srcDir, file), 'utf8');
    const min = await terserMinify(input, terserOptions);
    const obf = JavaScriptObfuscator.obfuscate(min.code || input, obfuscateOptions);
    await fs.writeFile(path.join(distDir, file), obf.getObfuscatedCode(), 'utf8');
  }
}

async function build(){
  await ensureCleanDist();
  await copyStatic();
  await Promise.all([
    processHTML(),
    processCSS(),
    processJS(),
  ]);
  console.log('Build completed -> dist/');
}

build().catch(err => { console.error(err); process.exit(1); });
